package gov.va.genisis2.service.impl;

import gov.va.genisis2.dao.IRequestDao;
import gov.va.genisis2.exceptions.GenisisServiceException;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.WorkflowStatus;
import gov.va.genisis2.service.IRequestService;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author 586338
 * @author Getaneh Kassahun
 *
 */
@Service
public class RequestService implements IRequestService {
	/**
	 * The LOGGER.
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(RequestService.class);
	
	@Autowired
	private IRequestDao requestDao;

	/**
	 * This method is used to create request.
	 * 
	 * @param request
	 *            The request.
	 * @return int This returnes created request id.
	 */
	@Override
	public int createRequest(Request request) throws GenisisServiceException {
		try {
			return requestDao.createRequest(request);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on createRequest", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getRequestsByUID.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returnes list of requestes.
	 */
	@Override
	public List<Request> getRequestsByUID(String uid) throws GenisisServiceException {
		try {
			return requestDao.getRequestsByUID(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestsByUID", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getRequestsById.
	 * 
	 * @param id
	 *            The id.
	 * @return Request This returns request.
	 */
	@Override
	public Request getRequestById(int id) throws GenisisServiceException {
		try {
			return requestDao.getRequestById(id);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestsById", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getAllRequests.
	 * 
	 * @return this returns list of requestes.
	 */
	@Override
	public List<Request> getAllRequests() throws GenisisServiceException {
		try {
			return requestDao.getAllRequests();
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getAllRequests", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getAllRequestsByStudyApproval.
	 * 
	 * @param studyApprovalId
	 *            The studyApprovalId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getAllRequestsByStudyApproval(int studyApprovalId) throws GenisisServiceException {
		try {
			return requestDao.getAllRequestsByStudyApproval(studyApprovalId);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getAllRequestsByStudyApproval", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	
	/**
	 * This method is used to getAllRequestsByStaus.
	 * 
	 * @param status
	 *            The status.
	 * @return this returns list of requets.
	 */
	@Override
	public List<Request> getAllRequestsByStaus(String status) throws GenisisServiceException {
		try {
			return requestDao.getAllRequestsByStaus(status);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getAllRequestsByStaus", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to submitOrModify entity.
	 * 
	 * @param request
	 *            The request.
	 * @return int This returns entity's id.
	 */
	@Override
	public int submitOrModify(Request request) throws GenisisServiceException {
		try {
			return requestDao.submitOrModify(request);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify Request", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	
	/**
	 * This method is used to submitOrModify RequestHistory.
	 * 
	 * @param requestHistory
	 *            The requestHistory.
	 * @return int This returns RequestHistory id.
	 * 
	 */
	@Override
	public int submitOrModify(RequestHistory requestHistory) throws GenisisServiceException {
		try {
			return requestDao.submitOrModify(requestHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception on submitOrModify RequestHistory;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getRequestHistoryById.
	 * 
	 * @param requestId
	 *            The requestId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<RequestHistory> getRequestHistoriesById(int requestId) throws GenisisServiceException {
		try {
			return requestDao.getRequestHistoriesById(requestId);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestHistoryById;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}



	/**
	 * This method is used to getRequestsByDataManager.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getRequestsByDataManagers(String uid) throws GenisisServiceException {
		try {
			return requestDao.getRequestsByDataManager(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestsByDataManagers;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getRequestsDataManagers.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requestes.
	 */
	@Override
	public List<Request> getRequestsDataManagers(String uid) throws GenisisServiceException {
		try {
			return requestDao.getRequestsDataManagers(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestsDataManagers;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	/**
	 * This method is used to getRequestsDataSourceManager.
	 * 
	 * @param uid
	 *            The uid.
	 * @return this returns list of requestes.
	 */
	@Override
	public List<Request> getRequestsDataSourceManagers(String uid) throws GenisisServiceException {
		try {
			return requestDao.getRequestsDataSourceManagers(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestsDataSourceManagers;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public List<RequestHistory> getRequestHistoriesByRequestIdAndStatus(int requestId, String status) throws GenisisServiceException {
		try {
			return requestDao.getRequestHistoriesByRequestIdAndStatus(requestId, status);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestHistoryByRequestIdAndStatus;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public List<RequestHistory> getRequestHistoriesWithCommentsById(int requestId) throws GenisisServiceException {
		try {
			return requestDao.getRequestHistoriesWithCommentsById(requestId);
		} catch (Exception ex) {
			LOGGER.error("Exception while querying getRequestHistoriesWithCommentsById;", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public List<WorkflowStatus> getRequestStatusCounts() throws GenisisServiceException {
		try {
			return requestDao.getRequestStatusCounts();
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getRequestStatusCounts.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

	@Override
	public WorkflowStatus getWorkflowStatus(String status) throws GenisisServiceException {
		try {
			return requestDao.getWorkflowStatus(status);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while querying getWorkflowStatus.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
	}

}
